/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/***********************************************************************
* File name     : target.c
************************************************************************
* Created by    : Joerg Weedermann, BID R&D, 05.06.96
* Last modified : ask clearcase
************************************************************************
* File contents :
* Target control functions
************************************************************************/

#include <typedefs.h>

#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <regconst.h>
#include <regx20.h>
#include <regx21.h>
#include <regx22.h>
#include <regx23.h>
#include <regx50.h>
#include <regx51.h>
#include <regx52.h>
#include <regx53.h>
#include <session.h>
#include <target.h>

/* TODO: remove this as soon as puprpget is in a good place */
#include <mini_api.h>


#define DEC_RESULT_BASE_SIZE_MISMATCH  0x01
#define DEC_RESULT_INVALID_SIZE        0x02
#define DEC_RESULT_INVALID_MODE        0x03
#define DEC_RESULT_CONFIG_ACCESS_ERROR 0x04
#define DEC_RESULT_HW_BASE_WRITE_ERROR 0x05
#define DEC_RESULT_INVALID_DECODER     0x06
#define DEC_RESULT_BASE_NOT_0          0x07
#define DEC_RESULT_ROM_SIZE_0_ENABLED  0x08
/* #define DEC_RESULT_HW_BUSY             0x09 ... not used */

/* Generic info for range checking */
static b_generic_infotype *GenInfo;
/* ------------------------------------------------------------------------
 * Target Behaviout Programming Functions
 * ------------------------------------------------------------------------ */

b_errtype EXPORT BestTargetAttrPageInit(b_handletype handle, b_int32 page_num)
{
  B_TRY_VARS_NO_PROG;
  b_int32 zw;
  b_int8 zw8, ZW8;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_TARGET_ATTR, &GenInfo));

    /* Check range of page_num */
    /* For E2926A the exact check is done in firmware ! */
    assert(GenInfo->num_pages >= 7);

    /* Check range of page_num page_num should be in range: E2925A: 1,..,255
     * E2926A: 1,..,7 or 1,..,63 depending on the AttibutePageSize */
    if (page_num == 0 || page_num > GenInfo->num_pages)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_TATTR_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }

    if (Best16BitRegisterFile(handle))
    {
      zw8 = (b_int8) page_num;
      (void) BestByteCopy(&ZW8, &zw8, 1UL);
      B_TRY(BestBasicCommand(handle, CMD_EXE_TATTR_PAGEINIT, &ZW8, IN_EXE_TATTR_PAGEINIT, NULL, NULL));
    }
    else
    {
      B_TRY(BestBasicBlockWrite(handle, TATTR_PROG_PAGE_NUM, (b_int8 *) & page_num, 4UL));
      B_TRY(BestBasicBlockWrite(handle, TATTR_CLEAR, (b_int8 *) & page_num, 1UL));
      zw = 0L;
      B_TRY(BestBasicBlockWrite(handle, TATTR_PROG_OFFSET, (b_int8 *) & zw, 4UL));
    }
  }
  B_ERRETURN(B_TRY_RET);
}



b_errtype EXPORT BestTargetAttrPtrSet(b_handletype handle, b_int32 page_num, b_int32 offset)
{
  B_TRY_VARS_NO_PROG;
  b_int8 zw, zw8[2];
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_TARGET_ATTR, &GenInfo));

    /* Range checking */
    /* Remark: For concatenated pages, the offset may be bigger than 0x1f in
     * E2926A */

    if (Best16BitRegisterFile(handle))
    {
      /* We only check against max-values here. Exact check is done in
       * firmware */
      assert(GenInfo->num_pages == 63);
      assert(GenInfo->num_blocks == 252);

      /* Check range of page_num, exact check is done in FW */
      if ((page_num == 0) || (page_num > GenInfo->num_pages))
      {
        BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
        BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_TATTR_PAGE);
        BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
        BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
        BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
        B_TRY_ERROR(B_E_RANGE);
      }
      /* Check range of offset */
      if (offset >= GenInfo->num_blocks)
      {
        BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
        BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_OFFSET);
        BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
        BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1);
        BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
        B_TRY_ERROR(B_E_RANGE);
      }

      zw = (b_int8) page_num;
      (void) BestByteCopy(zw8 + 0, &zw, 1UL);
      zw = (b_int8) offset;
      (void) BestByteCopy(zw8 + 1, &zw, 1UL);
      B_TRY(BestBasicCommand(handle, CMD_EXE_TATTR_PTRSET, zw8, IN_EXE_TATTR_PTRSET, NULL, NULL));
    }
    else
    {
      assert(GenInfo->num_pages == 255);
      assert(GenInfo->blocks_per_page == 32);

      /* Check range of page_num, exact check is done in FW */
      if ((page_num == 0) || (page_num > GenInfo->num_pages))
      {
        BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
        BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_TATTR_PAGE);
        BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
        BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
        BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
        B_TRY_ERROR(B_E_RANGE);
      }
      /* Check range of offset */
      if (offset >= GenInfo->blocks_per_page)
      {
        BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
        BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_OFFSET);
        BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
        BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->blocks_per_page - 1);
        BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
        B_TRY_ERROR(B_E_RANGE);
      }

      B_TRY(BestBasicBlockWrite(handle, TATTR_PROG_PAGE_NUM, (b_int8 *) & page_num, 4UL));
      B_TRY(BestBasicBlockWrite(handle, TATTR_PROG_OFFSET, (b_int8 *) & offset, 4UL));
    }
  }
  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 *
 * -------------------------------------------------------------------------- */

#define T_DSERR_BIT          0x00001000
#define T_WRPAR_BIT          0x00000800
#define T_APERR_BIT          0x00000400
#define T_DPERR_BIT          0x00000200
#define T_ABORT_BIT          0x00000100
#define T_DOLOOP_BIT         0x00000080
#define T_RETRY_BIT          0x00000040
#define T_DISC_BIT           0x00000020
#define T_WAITS_BITS         0x0000001F

b_errtype EXPORT BestTargetAttrPropSet(
    b_handletype handle,
    b_tattrproptype tattrprop,
    b_int32 value)
{
  B_DECLARE_FUNCNAME("BestTargetAttrPropSet [tattrprpset]");

  B_TRY_VARS_NO_PROG;
  b_int32 regval;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    /* dynamic capability checking concerning values of parameters */
    B_TRY(BestParamCheck(handle, B_PARAM_TARGET_ATTR, (b_int32) tattrprop, value));

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropLongSet(handle, CMD_EXE_TATTR_PROP_SET, (b_int8) tattrprop, value));
    }
    else
    {
      B_TRY(BestBasicBlockRead(handle, T_ATTR, (b_int8 *) & regval, 4UL));

      switch (tattrprop)
      {

      case B_T_WAITS:
        regval &= ~T_WAITS_BITS;/* clear all waits */
        regval |= (value & T_WAITS_BITS);
        break;

      case B_T_DOLOOP:
        if (value == 0)
        {
          regval &= ~T_DOLOOP_BIT;
        }
        else
        {
          regval |= T_DOLOOP_BIT;
        }
        break;

      case B_T_DPERR:
        if (value == 0)
        {
          regval &= ~T_DPERR_BIT;
        }
        else
        {
          regval |= T_DPERR_BIT;
        }
        break;

      case B_T_APERR:
        if (value == 0)
        {
          regval &= ~T_APERR_BIT;
        }
        else
        {
          regval |= T_APERR_BIT;
        }
        break;

      case B_T_WRPAR:
        if (value == 0)
        {
          regval &= ~T_WRPAR_BIT;
        }
        else
        {
          regval |= T_WRPAR_BIT;
        }
        break;

      case B_T_DSERR:
        if (value == 0)
        {
          regval &= ~T_DSERR_BIT;
        }
        else
        {
          regval |= T_DSERR_BIT;
        }
        break;

      case B_T_TERM:
        /* at first, clear all term bits */
        regval &= ~(T_ABORT_BIT | T_RETRY_BIT | T_DISC_BIT);
        switch (value)
        {
        case B_TERM_NOTERM:
          break;                /* already done */
        case B_TERM_RETRY:
          regval |= T_RETRY_BIT;
          break;
        case B_TERM_DISCONNECT:
          regval |= T_DISC_BIT;
          break;
        case B_TERM_ABORT:
          regval |= T_ABORT_BIT;
          break;
        default:
          B_FCT_PARAM_ERROR(3, "Invalid value");
        }
        break;

      default:
        B_FCT_PARAM_ERROR(2, "Invalid property");
      }                         /*lint !e788 */

      B_TRY(BestBasicBlockWrite(handle, T_ATTR, (b_int8 *) & regval, 4UL));
    }
  }
  B_ERRETURN(B_TRY_RET);
}



/* --------------------------------------------------------------------------
 * convenience function, programs one complete target attribute line
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTargetAllAttr1xProg(
    b_handletype handle,
    b_int32 doloop,
    b_int32 waits,
    b_int32 term,
    b_int32 dperr,
    b_int32 dserr,
    b_int32 aperr,
    b_int32 wrpar)
{
  B_DECLARE_FUNCNAME("BestTargetAllAttr1xProg []");

  B_TRY_VARS_NO_PROG;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    /* check term */
    switch (term)
    {
    case B_TERM_NOTERM:
    case B_TERM_RETRY:
    case B_TERM_ABORT:
    case B_TERM_DISCONNECT:
      break;

    default:
      B_FCT_PARAM_ERROR(4, "Invalid term");
    }

    /* E2926A has more properties, we set them to default */
    B_TRY(BestTargetAttrPropDefaultSet(handle));

    /* program block properties */
    B_TRY(BestTargetAttrPropSet(handle, B_T_DOLOOP, doloop));
    B_TRY(BestTargetAttrPropSet(handle, B_T_WAITS, waits));
    B_TRY(BestTargetAttrPropSet(handle, B_T_TERM, term));
    B_TRY(BestTargetAttrPropSet(handle, B_T_WRPAR, wrpar));

    B_TRY(BestTargetAttrPropSet(handle, B_T_DPERR, dperr));
    B_TRY(BestTargetAttrPropSet(handle, B_T_DSERR, dserr));
    B_TRY(BestTargetAttrPropSet(handle, B_T_APERR, aperr));

    /* program attr line */
    B_TRY(BestTargetAttrPhaseProg(handle));
  }
  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestTargetAttrPropGet(b_handletype handle,
    b_tattrproptype tattrprop,
    b_int32 * value)
{
  B_DECLARE_FUNCNAME("BestTargetAttrPropGet [taprpget]");

  B_TRY_VARS_NO_PROG;
  b_int32 regval;
  b_param_infotype *ParamInfo;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    B_FCT_PARAM_NULL_POINTER_CHK(value);

    /* Check range of tattrprop */
    B_TRY(BestParamInfoGet(handle, B_PARAM_TARGET_ATTR, (b_int32) tattrprop,
        NULL, (b_int32) B_ENUM_SEARCH));

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestParamInfoGet(handle, B_PARAM_TARGET_ATTR, (b_int32) tattrprop,
          &ParamInfo, (b_int32) B_ENUM_SEARCH));

      /* Get current value */
      B_TRY(BestAbstractPropLongGet(handle, CMD_EXE_TATTR_PROP_GET,
          (b_int8) ParamInfo->proptyp.tattrprop, value));
    }
    else
    {
      B_TRY(BestBasicBlockRead(handle, T_ATTR, (b_int8 *) & regval, 4UL));

      switch (tattrprop)
      {

      case B_T_WAITS:
        *value = (regval & T_WAITS_BITS);
        break;

      case B_T_DOLOOP:
        *value = ((regval & T_DOLOOP_BIT) == 0) ? 0 : 1;
        break;

      case B_T_DPERR:
        *value = ((regval & T_DPERR_BIT) == 0) ? 0 : 1;
        break;

      case B_T_DSERR:
        *value = ((regval & T_DSERR_BIT) == 0) ? 0 : 1;
        break;

      case B_T_APERR:
        *value = ((regval & T_APERR_BIT) == 0) ? 0 : 1;
        break;

      case B_T_WRPAR:
        *value = ((regval & T_WRPAR_BIT) == 0) ? 0 : 1;
        break;

      case B_T_TERM:
        if ((regval & T_RETRY_BIT) != 0)
        {
          *value = B_TERM_RETRY;
          break;
        }
        if ((regval & T_DISC_BIT) != 0)
        {
          *value = B_TERM_DISCONNECT;
          break;
        }
        if ((regval & T_ABORT_BIT) != 0)
        {
          *value = B_TERM_ABORT;
          break;
        }
        *value = B_TERM_NOTERM;
        break;

      default:
        B_FCT_PARAM_ERROR(2, "Invalid property");
      }                         /*lint !e788 */
    }
  }
  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestTargetAttrPropDefaultSet(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  b_int32 zw;
  b_int32 i;
  b_param_infotype *ParamInfo;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    if (Best16BitRegisterFile(handle))
    {
      /* Get pointer to generic info */
      B_TRY(BestGenInfoGet(handle, B_PARAM_TARGET_ATTR, &GenInfo));

      for (i = 0; i < GenInfo->num_elem; i++)
      {
        /* Get pointer to i-th (existing) property */
        B_TRY(BestParamInfoGet(handle, B_PARAM_TARGET_ATTR, i, &ParamInfo,
            (b_int32) B_INDEX_SEARCH));

        /* Set it to default */
        B_TRY(BestTargetAttrPropSet(handle, ParamInfo->proptyp.tattrprop, ParamInfo->defaultval));
      }
    }
    else
    {
      zw = 0L;
      B_TRY(BestBasicBlockWrite(handle, T_ATTR, (b_int8 *) & zw, 4UL));
    }
  }
  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestTargetAttrPhaseProg(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  b_int16 attrib;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_EXE_TATTR_PHASEPROG, NULL, 0, NULL, NULL));
    }
    else
    {
      attrib = TARGET_PHASE_PROG;
      B_TRY(BestBasicBlockWrite(handle, PCI_COMMAND, (b_int8 *) & attrib, 2UL));
    }
  }
  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestTargetAttrPhaseRead(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  b_int16 attrib;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_EXE_TATTR_PHASEREAD, NULL, 0, NULL, NULL));
    }
    else
    {
      attrib = TARGET_PHASE_READ;
      B_TRY(BestBasicBlockWrite(handle, PCI_COMMAND, (b_int8 *) & attrib, 2UL));
    }
  }
  B_ERRETURN(B_TRY_RET);
}


/* ------------------------------------------------------------------------
 * Exerciser Run Functions
 * ------------------------------------------------------------------------ */

b_errtype EXPORT BestTargetAttrPageSelect(b_handletype handle, b_int32 page_num)
{
  B_TRY_VARS_NO_PROG;
  b_int8 zw8, zw;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_TARGET_ATTR, &GenInfo));

    /* Check range of page_num For E2926A the exact check is done in firmware
     * ! page_num should be in range: E2925A: 0,..,255 E2926A: 0,..,7 or
     * 1,..,63 depending on the AttibutePageSize */
    if (page_num > GenInfo->num_pages)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_TATTR_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }

    if (Best16BitRegisterFile(handle))
    {
      zw8 = (b_int8) page_num;
      (void) BestByteCopy(&zw, &zw8, 1UL);
      B_TRY(BestBasicCommand(handle, CMD_EXE_TATTR_PAGESELECT, &zw, IN_EXE_TATTR_PAGESELECT, NULL, NULL));
    }
    else
    {
      B_TRY(BestBasicBlockWrite(handle, TATTR_RUN_PAGE_NUM, (b_int8 *) & page_num, 4UL));
    }
  }
  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestTargetAttrLineProg(b_handletype handle, b_int32 page_num, b_int32 offset)
{
  B_TRY_VARS_NO_PROG;
  b_errtype err;

  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_TARGET_ATTR, &GenInfo));

    /* Check range of page_num, exact check is done in FW */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_TATTR_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of offset */
    if (offset >= GenInfo->num_blocks)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }

    if (Best16BitRegisterFile(handle))
    {
      b_int8 page8, offset8, zw8[2];
      page8 = (b_int8) page_num;
      offset8 = (b_int8) offset;

      (void) BestByteCopy(zw8 + 0, &page8, 1UL);
      (void) BestByteCopy(zw8 + 1, &offset8, 1UL);

      B_TRY(BestBasicCommand(handle, CMD_EXE_TATTR_LINEPROG, zw8, IN_EXE_TATTR_LINEPROG, NULL, NULL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* programming properties of one group only */
b_errtype EXPORT BestTargetAttrGroupLineProg(b_handletype handle, b_tattrgrouptype group,
    b_int32 page_num, b_int32 offset)
{
  B_TRY_VARS_NO_PROG;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_TARGET_ATTR, &GenInfo));

    /* Check range of page_num, exact check is done in FW */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_TATTR_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of offset */
    if (offset >= GenInfo->num_blocks)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of group */
    if (group >= B_TATTR_GRP_NUM)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_GROUP);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, (b_int32) B_TATTR_GRP_NUM - 1);
      BestLastErrorParamSet(handle, B_ERRPAR_5, (b_int32) group);
      B_TRY_ERROR(B_E_RANGE);
    }

    if (Best16BitRegisterFile(handle))
    {
      b_int8 group8, page8, offset8, zw8[3];
      group8 = (b_int8) group;
      page8 = (b_int8) page_num;
      offset8 = (b_int8) offset;

      (void) BestByteCopy(zw8 + 0, &group8, 1UL);
      (void) BestByteCopy(zw8 + 1, &page8, 1UL);
      (void) BestByteCopy(zw8 + 2, &offset8, 1UL);

      B_TRY(BestBasicCommand(handle, CMD_EXE_TATTR_GRPLINEPROG, zw8, IN_EXE_TATTR_GRPLINEPROG, NULL, NULL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestTargetAttrLineRead(b_handletype handle, b_int32 page_num, b_int32 offset)
{
  B_TRY_VARS_NO_PROG;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_TARGET_ATTR, &GenInfo));

    /* Check range of page_num, exact check is done in FW */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_TATTR_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of offset */
    if (offset >= GenInfo->num_blocks)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }

    if (Best16BitRegisterFile(handle))
    {
      b_int8 page8, offset8, zw8[2];
      page8 = (b_int8) page_num;
      offset8 = (b_int8) offset;

      (void) BestByteCopy(zw8 + 0, &page8, 1UL);
      (void) BestByteCopy(zw8 + 1, &offset8, 1UL);

      B_TRY(BestBasicCommand(handle, CMD_EXE_TATTR_LINEREAD, zw8, IN_EXE_TATTR_LINEREAD, NULL, NULL));
    }
  }
  B_ERRETURN(B_TRY_RET);
}


/* reading properties of one group only */
b_errtype EXPORT BestTargetAttrGroupLineRead(b_handletype handle, b_tattrgrouptype group,
    b_int32 page_num, b_int32 offset)
{
  B_TRY_VARS_NO_PROG;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_TARGET_ATTR, &GenInfo));

    /* Check range of page_num, exact check is done in FW */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_TATTR_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of offset */
    if (offset >= GenInfo->num_blocks)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of group */
    if (group >= B_TATTR_GRP_NUM)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) B_PCST_GROUP);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, (b_int32) B_TATTR_GRP_NUM - 1);
      BestLastErrorParamSet(handle, B_ERRPAR_5, (b_int32) group);
      B_TRY_ERROR(B_E_RANGE);
    }

    if (Best16BitRegisterFile(handle))
    {
      b_int8 group8, page8, offset8, zw8[3];
      group8 = (b_int8) group;
      page8 = (b_int8) page_num;
      offset8 = (b_int8) offset;

      (void) BestByteCopy(zw8 + 0, &group8, 1UL);
      (void) BestByteCopy(zw8 + 1, &page8, 1UL);
      (void) BestByteCopy(zw8 + 2, &offset8, 1UL);

      B_TRY(BestBasicCommand(handle, CMD_EXE_TATTR_GRPLINEREAD, zw8, IN_EXE_TATTR_GRPLINEREAD, NULL, NULL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* ------------------------------------------------------------------------
 * Target Generic Propertie functions
 * ------------------------------------------------------------------------ */

/* function to set generic property */
b_errtype EXPORT BestTargetGenPropSet(
    b_handletype handle,
    b_targetgenproptype targetgenprop,
    b_int32 value)
{
  B_DECLARE_FUNCNAME("BestTargetGenPropSet [tgenprpset]");

  B_TRY_VARS_NO_PROG;
  b_errtype err;
  b_int32 zw;
  b_int8 zw8;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_HOSTINT);

    /* dynamic capability checking concerning values of parameters */
    B_TRY(BestParamCheck(handle, B_PARAM_TARGET_GEN,
        (b_int32) targetgenprop, value));

    if (Best16BitRegisterFile(handle))
    {
      /* E2926 */
      B_TRY(BestAbstractPropLongSet(handle, CMD_EXEFHIF_TGEN_PROP_SET, (b_int8) targetgenprop, value));
    }
    else
    {
      /* E2925 */
      switch (targetgenprop)
      {

      case B_TGEN_MEMSPACE:
        zw = 0x00000004;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS, (b_int8 *) & zw, 4UL));
        B_TRY(BestBasicBlockRead(handle, CONF_SPACE_DATA, (b_int8 *) & zw, 4UL));
        if (value == 0)
          zw &= 0xfffffffdUL;
        else
          zw |= 0x00000002;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_DATA, (b_int8 *) & zw, 4UL));
        break;

      case B_TGEN_IOSPACE:
        zw = 0x00000004;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS, (b_int8 *) & zw, 4UL));
        B_TRY(BestBasicBlockRead(handle, CONF_SPACE_DATA, (b_int8 *) & zw, 4UL));
        if (value == 0)
          zw &= 0xfffffffeUL;
        else
          zw |= 0x00000001;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_DATA, (b_int8 *) & zw, 4UL));
        break;

      case B_TGEN_ROMENABLE:
        zw = 0x00000030;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS, (b_int8 *) & zw, 4UL));
        B_TRY(BestBasicBlockRead(handle, CONF_SPACE_DATA, (b_int8 *) & zw, 4UL));
        if (value == 0)
          zw &= 0xfffffffeUL;
        else
          zw |= 0x00000001;
        err = BestBasicBlockWrite(handle, CONF_SPACE_DATA, (b_int8 *) & zw, 4UL);
        if (err == B_E_FUNC)
        {                       /* an error occured, so get the result code */
          B_TRY(BestBasicBlockRead(handle, T_DEC_RESULT, &zw8, 1UL));
          switch (zw8)
          {
          case DEC_RESULT_CONFIG_ACCESS_ERROR:
            B_TRY_ERROR(B_E_DEC_CONFIG_ACCESS);

          case DEC_RESULT_HW_BASE_WRITE_ERROR:
            B_TRY_ERROR(B_E_DEC_BASE_WRITE);

          case DEC_RESULT_BASE_SIZE_MISMATCH:
            B_TRY_ERROR(B_E_DEC_SIZE_BASE_MISMATCH);

          case DEC_RESULT_INVALID_SIZE:
            B_TRY_ERROR(B_E_DEC_INVALID_SIZE);

          case DEC_RESULT_INVALID_MODE:
            B_TRY_ERROR(B_E_DEC_INVALID_MODE);

          case DEC_RESULT_INVALID_DECODER:
            B_TRY_ERROR(B_E_DEC_INVALID_DECODER);

          case DEC_RESULT_BASE_NOT_0:
            B_TRY_ERROR(B_E_DEC_BASE_NOT_0);

          case DEC_RESULT_ROM_SIZE_0_ENABLED:
            B_TRY_ERROR(B_E_DEC_ROM_SIZE_0_ENABLED);

          default:
            B_TRY_ERROR(B_E_FUNC);
          }
        }
        break;

      case B_TGEN_RUNMODE:
        zw8 = (b_int8) value;
        B_FCT_PARAM_CHK(3,
          ((value != B_RUNMODE_ADDRRESTART) && (value != B_RUNMODE_SEQUENTIAL)));

        B_TRY(BestBasicBlockWrite(handle, TGEN_RUNMODE, &zw8, 1UL));
        break;

      default:
        B_FCT_PARAM_ERROR(2, "Invalid property");
      }  /* CONSIDER; B_TGEN_BACKCAPABLE not used */
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* function delivers a specific generic property */
b_errtype EXPORT BestTargetGenPropGet(
    b_handletype handle,
    b_targetgenproptype targetgenprop,
    b_int32 * value)
{
  B_DECLARE_FUNCNAME("BestTargetGenPropGet [tgenprpget]");

  B_TRY_VARS_NO_PROG;
  b_int32 zw, BitPos;
  b_int8 zw8;
  b_param_infotype *ParamInfo;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_HOSTINT);

    B_FCT_PARAM_NULL_POINTER_CHK(value);

    if (!BestIs2925(handle))
    {
      /* E2926 */
      switch (targetgenprop)
      {
      case B_TGEN_MEMSPACE:
      case B_TGEN_IOSPACE:
        {
          /* These two props belong to the config-space and thus are not held
           * in FW ! Read Command register from config-space: read bit 0 or 1
           * in config command register */
          B_TRY(BestConfRegGet(handle, 0x4UL, &zw));
          BitPos = (targetgenprop == B_TGEN_IOSPACE ? 0UL : 1UL);
          *value = (zw & (1UL << (int) BitPos) ? 1UL : 0UL);
        }
        break;

      case B_TGEN_ROMENABLE:
        {
          /* This prop belongs to the config-space and thus is not held in FW
           * ! Read register from config-space: read bit 0 in config
           * expansion ROM base address register */
          B_TRY(BestConfRegGet(handle, 0x30UL, &zw));
          *value = (zw & 1UL ? 1UL : 0UL);
        }
        break;

      case B_TGEN_BACKCAPABLE:
        {
          /* This prop belongs to the config-space and thus is not held in FW
           * ! Read register from config-space: read bit 7 in config status
           * register */
          B_TRY(BestConfRegGet(handle, 0x4UL, &zw));
          *value = (zw & 0x800000UL ? 1UL : 0UL);
        }
        break;

      default:
        {
          /* Get pointer to i-th (existing) property */
          B_TRY(BestParamInfoGet(handle, B_PARAM_TARGET_GEN,
              (b_int32) targetgenprop, &ParamInfo, (b_int32) B_ENUM_SEARCH));
          /* Get current value */
          B_TRY(BestAbstractPropLongGet(handle, CMD_EXEFHIF_TGEN_PROP_GET,
              (b_int8) ParamInfo->proptyp.tgenprop, value));
	}
      }  /* switch */  /* CONSIDER; B_TGEN_RUNMODE not used */
    }
    else
    {
      /* E2925 */
      switch (targetgenprop)
      {

      case B_TGEN_RUNMODE:
        B_TRY(BestBasicBlockRead(handle, TGEN_RUNMODE, &zw8, 1UL));
        *value = (b_int32) zw8;
        break;

      case B_TGEN_MEMSPACE:
        zw = 0x00000004;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS, (b_int8 *) & zw, 4UL));
        B_TRY(BestBasicBlockRead(handle, CONF_SPACE_DATA, (b_int8 *) & zw, 4UL));
        *value = (zw & 2UL ? 1UL : 0UL);
        break;

      case B_TGEN_IOSPACE:
        zw = 0x00000004;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS, (b_int8 *) & zw, 4UL));
        B_TRY(BestBasicBlockRead(handle, CONF_SPACE_DATA, (b_int8 *) & zw, 4UL));
        *value = (zw & 1 ? 1 : 0);
        break;

      case B_TGEN_ROMENABLE:
        zw = 0x00000030;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS, (b_int8 *) & zw, 4UL));
        B_TRY(BestBasicBlockRead(handle, CONF_SPACE_DATA, (b_int8 *) & zw, 4UL));
        *value = (zw & 1 ? 1 : 0);
        break;

      default:
        B_FCT_PARAM_ERROR(2, "Invalid property");
      }  /* CONSIDER; B_TGEN_BACKCAPABLE not used */
    }
  }
  B_ERRETURN(B_TRY_RET);
}


/* function sets all generic properties to their default values */
b_errtype EXPORT BestTargetGenPropDefaultSet(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  b_int32 i, conf_restore;
  b_param_infotype *ParamInfo;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_TARGET_GEN, &GenInfo));
    B_TRY(BestPowerUpPropGet(handle, B_PU_CONFRESTORE, &conf_restore));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      B_TRY(BestParamInfoGet(handle, B_PARAM_TARGET_GEN,
          i, &ParamInfo, (b_int32) B_INDEX_SEARCH));

      switch (ParamInfo->proptyp.tgenprop)
      {
      case B_TGEN_IOSPACE:
      case B_TGEN_MEMSPACE:
      case B_TGEN_ROMENABLE:
      case B_TGEN_BACKCAPABLE:
        if (conf_restore == 0)
          break;

      default:                  /*lint !e616 .. deliberate flow into default */
        /* Set it to default */
        B_TRY(BestTargetGenPropSet(handle, ParamInfo->proptyp.tgenprop,
            ParamInfo->defaultval));
        break;
      }  /* CONSIDER; B_TGEN_RUNMODE not used */
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* -----------------------------------------------------------------
 * Expansion ROM Programming Functions
 * ----------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * Write 1st byte of value at offset
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestExpRomByteWrite(
    b_handletype handle,
    b_int32 offset,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestExpRomByteWrite [erbytewrite]");

  B_TRY_VARS_NO_PROG;
  b_int8 value8 = (b_int8) value;
  b_int32 zw32;
  B_FCT_PARAM_CHK(1, offset > 0xffffUL);
  B_FCT_PARAM_CHK(2, value > 0xffUL);

  B_TRY_BEGIN
  {
    B_TRY(BestBoardPropGet(handle, B_BOARD_ROMUSAGE, &zw32));
    B_TRY_FAIL(zw32 == B_ROMUSAGE_INTERNAL ? B_E_ROM_INTERNAL : B_E_OK);

    /* send offset and byte */
    if (Best16BitRegisterFile(handle))
    {
      b_int8 cmdbuf[6];
      b_int8ptr p_cmdbuf = cmdbuf;
      p_cmdbuf = BestLong2Stream(p_cmdbuf, &offset, 1UL);
      p_cmdbuf = BestByteCopy(p_cmdbuf, &value8, 1UL);

      B_TRY(BestBasicCommand(handle, CMD_EXP_ROM_BYTE_WRITE,
          cmdbuf, IN_EXP_ROM_BYTE_WRITE, NULL, NULL));
    }
    else
    {
      B_TRY(BestBasicBlockWrite(handle, EXP_ROM_OFFS, (b_int8 *) & offset, 4UL));
      B_TRY(BestBasicBlockWrite(handle, EXP_ROM_DATA, &value8, 1UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * Read one byte at offset into *value
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestExpRomByteRead(
    b_handletype handle,
    b_int32 offset,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestExpRomByteRead [erbyteread]");

  b_errtype err;
  b_int8 byteval;
  B_FCT_PARAM_CHK(1, offset > 0xffff);
  B_FCT_PARAM_NULL_POINTER_CHK(value);

  if (Best16BitRegisterFile(handle))
  {
    b_int8 cmdbuf[4];
    b_int16 outsize = OUT_EXP_ROM_BYTE_READ;
    (void) BestLong2Stream(cmdbuf, &offset, 1UL);

    err = BestBasicCommand(handle, CMD_EXP_ROM_BYTE_READ,
      cmdbuf, IN_EXP_ROM_BYTE_READ, cmdbuf, &outsize);

    (void) BestByteCopy(&byteval, cmdbuf, 1UL);

  }
  else
  {
    B_ERRCHECK(BestBasicBlockWrite(handle, EXP_ROM_OFFS, (b_int8 *) & offset, 4UL));

    err = BestBasicBlockRead(handle, EXP_ROM_DATA, &byteval, 1UL);
  }

  *value = (b_int32) byteval;
  B_ERRETURN(err);
}
